/** @file   PlayerCar.cpp
 * @brief   Implementation of PlayerCar - class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/08/13 10:50:22 $
 * @author  Tomi Lamminsaari
 */


#include "PlayerCar.h"
#include <fstream>
#include "www_exceptions.h"
#include "settings.h"
#include "gfxid.h"
#include "www_assert.h"
#include "redrawqueue.h"
#include "playercontroller.h"
#include "carcontroller.h"
#include "player.h"
#include "warglobals.h"
#include "www_map.h"
#include "utils.h"
#include "bullettable.h"
#include "soundsamples.h"
#include "animplayer.h"
#include "gameanims.h"
#include "GfxManager.h"
#include "AnimId.h"
using std::ifstream;
using std::string;
using namespace eng2d;

namespace WeWantWar {



///
/// Static data, datatypes and constants
/// =================================================================




///
/// Static methods
/// =================================================================



///
/// Constructors, destructor and operators
/// =================================================================

/** Constructor.
 */
PlayerCar::PlayerCar( const string& aRouteName ) :
  PolygonObject(),
  iMaster( 0 ),
  iGunAngle( 0 ),
  iGunGraphics( 0 ),
  iGunDelay( 2 ),
  iVelocity( 0 )
{
  // Load the polygon data
  LOG_MESSAGE( "PlayerCar(): Opening file 'data/polygons/transit.txt'" );
  ifstream fin( "data/polygons/transit.txt" );
  if ( !fin ) {
    LOG_MESSAGE( "PlayerCar(): Failed to open polygon file" );
    alert( "Failed to read polygon",
           "data/polygons/transit.txt", 0,
           "ok", 0, 0, 0 );
    throw Exception( "CarPlayer() failed" );
    
  } else {
    // Read the opening tag from the file and start reading the object.
    LOG_MESSAGE( "PlayerCar(): File opened. Starting to read from it..." );
    string tmp;
    fin >> tmp;
    int ret = this->read( fin );
    fin.close();
    if ( ret != 0 ) {
      alert( "Failed to read polygon",
             "data/polygons/transit.txt", 0, "ok", 0, 0, 0 );
      throw Exception( "CarPlayer() failed" );
    }
  }
  LOG_MESSAGE( "PlayerCar(): Polygonfile read" );
  // Add the textures needed for this object.
//  this->addTexture( GfxPool::texture_bitmaps.at( GfxPool::TEX_PLAYERCAR ) );
  
  // Take the object settings from cofiguration file.
  LOG_MESSAGE( "PlayerCar(): Requesting settings" );
  ObjectID::Type oid = ObjectID::TYPE_PLAYERCAR;
  this->setArmor( Settings::floatObjProp(oid, "playercar_armor:") );
  this->boundingSphere( Settings::floatObjProp(oid, "playercar_bsphere:") );
  iVelocity = Settings::floatObjProp(oid, "playercar_spd:");
  iGunDelay = Settings::intObjProp(oid, "playercar_rdelay:");
  
  // Find the machinegun graphics
  LOG_MESSAGE( "PlayerCar(): Requesting graphics" );
  iGunGraphics = GfxManager::findBitmap( GfxId::KPlayerGun );
  
  // Controlpoint for machinegun
  this->addCtrlPoint( Vec2D(0, 20) );
  // Set the relative position of the gun nose
  iGunNosePos.set(0, -13);
  
  // Controller
  Route route = WarGlobals::pLevel->getRoute( aRouteName );
  CarController* controller = new CarController(this);
  controller->setRoute( route );
  this->setController( controller );
}



/** Destructor.
 */
PlayerCar::~PlayerCar()
{
}


/** Updates this object
 */
void PlayerCar::update()
{
  BaseController* controller = this->getController();
  controller->update();
  if ( controller->forward() == 2 ) {
    Vec2D mvec( 0, -iVelocity );
    mvec.rotate( this->angle() );
    this->move( mvec );
    
  } else if ( controller->forward() == 1 ) {
    Vec2D mvec( 0, -iVelocity );
    mvec /= 2.5;
    mvec.rotate( this->angle() );
    this->move( mvec );
    
  }
  this->changeAngle( controller->turn() * 2 );
  
  if ( iMaster != 0 ) {
    this->updateCaptured();
  }
}



/** Redraws this object
 */
void PlayerCar::redraw( RedrawQueue* aQueue )
{
  PolygonObject::redraw( aQueue );
  
  Vec2D pos = this->getCtrlPoint( KMachinegunCtrlPointIndex );
  pos += this->position();
  int gunX = pos.intX() - Map::scrollX;
  int gunY = pos.intY() - Map::scrollY;
  gunX -= iGunGraphics->w / 2;
  gunY -= iGunGraphics->h / 2;
  aQueue->addRotatedSprite( RedrawQueue::PRI_ABOVE_NORMAL, gunX,gunY,
                            itofix(iGunAngle), iGunGraphics );
}



/** Makes sound
 */
void PlayerCar::makeSound( GameObject::SoundID aSoundId ) const
{
  switch ( aSoundId ) {
    case ( GameObject::SND_ATTACK ): {
      Weapon weapon( Weapon::W_MINIGUN );
      Sound::playSample( weapon.getSoundID(), false );
      break;
    }
    default: {
      break;
    }
  }
}



/** Destroys this car.
 */
void PlayerCar::kill()
{
}



/** Manages the bullet hits
 */
bool PlayerCar::hitByBullet( Bullet* aBullet )
{
  if ( aBullet->iOwner != 0 ) {
    if ( aBullet->iOwner->objectType() == ObjectID::TYPE_PLAYER ) {
      // Player can not shoot us.
      return false;
    }
  }
  return true;
}



/** Captures/releases this object
 */
void PlayerCar::capture( GameObject* aMaster )
{
  if ( iMaster != 0 && aMaster == 0 ) {
    // Tell the previous master that he no longer is our master.
    iMaster->state( GameObject::STATE_LIVING );
    iMaster->hidden( false );
    Player* player = dynamic_cast<Player*>( iMaster );
    if ( player != 0 ) {
      // Our master was player
      player->setVehicle( 0 );
    }
    iMaster = 0;
    return;
  }
  
  // Inform the new master that he is accepted to be our master.
  iMaster = aMaster;
  iMaster->state( GameObject::STATE_HIBERNATING );
  iMaster->hidden( true );
  iMaster->position( this->position() );
  Player* player = dynamic_cast<Player*>( iMaster );
  if ( player != 0 ) {
    // Our new master is player.
    player->setVehicle( this );
  }
  this->setCounter( KNoExitCounterIndex, 60 );
}



/** Resurrects this object.
 */
void PlayerCar::resurrect()
{
}


/** Tells if this object can be captured.
 */
bool PlayerCar::canBeCaptured() const
{
  if ( iMaster == 0 ) {
    return true;
  }
  return false;
}



/** Returns the pointer to master object.
 */
GameObject* PlayerCar::getMaster() const
{
  return iMaster;
}



/** Are we in reloading state
 */
bool PlayerCar::reloading() const
{
  if ( this->getCounter( KReloadingCounterIndex ) > 0 ) {
    return true;
  }
  return false;
}



/** Returns the type of this object.
 */
ObjectID::Type PlayerCar::objectType() const
{
  return ObjectID::TYPE_PLAYERCAR;
}


/** Updates the captured car
 */
void PlayerCar::updateCaptured()
{
  if ( iMaster->getController() == 0 ) {
    return;
  }
  if ( iMaster->objectType() != ObjectID::TYPE_PLAYER ) {
    // Player is not our master. This feature has not been implemented
    // yet.
    return;
  }
  
  PlayerController* controller =
      dynamic_cast<PlayerController*>( iMaster->getController() );

  // Update the controller and move the car.
  controller->update();
  
  if ( controller->shoot() != 0 ) {
    this->shoot();
  }
  
  // Update the mouse cursor and rotate the machine gun.
  Vec2D mousePos( mouse_x + Map::scrollX, mouse_y + Map::scrollY );
  Vec2D gunPos( this->getCtrlPoint( KMachinegunCtrlPointIndex ) );
  gunPos += this->position();
  int gunAngleChange = Utils::findTurningDir( gunPos, mousePos, iGunAngle );
  this->rotateMachineGun( gunAngleChange * 2 );
  
  // Set the mouse cursor position so that the crosshair will be drawn in
  // right position.
  mousePos.set( mouse_x, mouse_y );
  WarGlobals::pHud->crosshairPos( mousePos );
  
  // Update the position of our master.
  iMaster->position( this->position() );
  if ( this->getCounter( KNoExitCounterIndex ) < 0 ) {
    if ( key[KEY_E] ) {
      WarGlobals::captureVehicle = true;
    }
  }
}



/** Rotates the machine gun.
 */
void PlayerCar::rotateMachineGun( int aAngleChange )
{
  iGunAngle += aAngleChange;
  iGunAngle &= 255;
}



/** Shoots
 */
void PlayerCar::shoot()
{
  if ( this->reloading() == true ) {
    return;
  }
  
  // Calculate the position of the gun nose.
  Vec2D nosePos( iGunNosePos );
  nosePos.rotate( iGunAngle );
  nosePos += this->position();
  nosePos += this->getCtrlPoint( KMachinegunCtrlPointIndex );
  
  // Create the bullet
  Bullet* bullet = BulletTable::createBullet( iMaster, nosePos, Bullet::EMinigun );
  Vec2D spdVec( 0, -( bullet->velocity().length() ) );
  spdVec.rotate( iGunAngle );
  bullet->setVelocity( spdVec );
  WarGlobals::pBulletManager->spawnBullet( bullet );
  
  Weapon::Specs& weapon = Weapon::getWeaponSpecs( Weapon::W_MINIGUN );
  this->setCounter( KReloadingCounterIndex, weapon.reload + 1 );

  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KMinigunShootFlame );
  AnimPlayer::spawn( flameAnim, nosePos, 0 );
  this->makeSound( GameObject::SND_ATTACK );
  
}

}  // end of namespace
